// YouTube Dubbing Extension - Background Service Worker
// Handles API communication with backend
// ToolWeb.in

const API_BASE_URL = 'https://api.toolweb.in:8161/dubby';

// Store active jobs
const activeJobs = new Map();

// Message listener
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  switch (message.action) {
    case 'startDubbing':
      handleStartDubbing(message.settings, sendResponse);
      return true; // Keep channel open
    
    case 'checkProgress':
      handleCheckProgress(message.jobId, sendResponse);
      return true;
    
    case 'cancelDubbing':
      handleCancelDubbing(message.jobId, sendResponse);
      return true;
    
    default:
      sendResponse({ success: false, error: 'Unknown action' });
  }
});

async function handleStartDubbing(settings, sendResponse) {
  try {
    console.log('[Background] Starting dubbing with settings:', settings);
    
    // Make API request to start dubbing job
    const response = await fetch(`${API_BASE_URL}/start`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({
        video_id: settings.videoId,
        target_language: settings.targetLanguage,
        voice_gender: settings.voiceGender,
        voice_speed: settings.voiceSpeed,
        original_volume: settings.originalVolume
      })
    });
    
    if (!response.ok) {
      const error = await response.json();
      throw new Error(error.detail || 'API request failed');
    }
    
    const data = await response.json();
    
    // Store job info
    activeJobs.set(data.job_id, {
      videoId: settings.videoId,
      status: 'processing',
      progress: 0
    });
    
    sendResponse({
      success: true,
      jobId: data.job_id
    });
    
  } catch (error) {
    console.error('[Background] Start dubbing error:', error);
    sendResponse({
      success: false,
      error: error.message || 'Failed to start dubbing'
    });
  }
}

async function handleCheckProgress(jobId, sendResponse) {
  try {
    const response = await fetch(`${API_BASE_URL}/status/${jobId}`);
    
    if (!response.ok) {
      throw new Error('Failed to check progress');
    }
    
    const data = await response.json();
    
    // Update stored job
    if (activeJobs.has(jobId)) {
      activeJobs.get(jobId).status = data.status;
      activeJobs.get(jobId).progress = data.progress;
    }
    
    if (data.status === 'completed') {
      // Return the dubbed audio data
      sendResponse({
        status: 'completed',
        audioData: data.audio_data,
        segments: data.segments
      });
      
      // Clean up
      activeJobs.delete(jobId);
      
    } else if (data.status === 'error') {
      sendResponse({
        status: 'error',
        error: data.error || 'Processing failed'
      });
      activeJobs.delete(jobId);
      
    } else {
      sendResponse({
        status: 'processing',
        progress: data.progress,
        message: data.message || 'Processing...'
      });
    }
    
  } catch (error) {
    console.error('[Background] Check progress error:', error);
    sendResponse({
      status: 'error',
      error: error.message
    });
  }
}

async function handleCancelDubbing(jobId, sendResponse) {
  try {
    await fetch(`${API_BASE_URL}/cancel/${jobId}`, {
      method: 'POST'
    });
    
    activeJobs.delete(jobId);
    
    sendResponse({ success: true });
  } catch (error) {
    console.error('[Background] Cancel error:', error);
    sendResponse({ success: false, error: error.message });
  }
}

// Handle extension install/update
chrome.runtime.onInstalled.addListener((details) => {
  if (details.reason === 'install') {
    console.log('[YouTube Dubbing] Extension installed');
    
    // Set default settings
    chrome.storage.local.set({
      targetLanguage: 'hi',
      voiceGender: 'female',
      voiceSpeed: '1',
      originalVolume: '20'
    });
  }
});
