// YouTube Dubbing Extension - Content Script
// Injected into YouTube pages
// ToolWeb.in

(function() {
  'use strict';

  // State
  let dubbedAudio = null;
  let audioContext = null;
  let gainNode = null;
  let sourceNode = null;
  let isDubbing = false;
  let segments = [];
  let currentSegmentIndex = 0;
  let syncInterval = null;
  let videoElement = null;
  let originalVolume = 0.2;
  let initAttempts = 0;
  const MAX_INIT_ATTEMPTS = 10;

  // Initialize when page loads
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }

  function init() {
    // Find the video element
    findVideoElement();
    
    // Watch for page navigation (YouTube SPA)
    observeNavigation();
    
    console.log('[YouTube Dubbing] Content script initialized');
  }

  function findVideoElement() {
    // Try multiple selectors for YouTube video
    videoElement = document.querySelector('video.html5-main-video') || 
                   document.querySelector('video.video-stream') ||
                   document.querySelector('#movie_player video') ||
                   document.querySelector('ytd-player video') ||
                   document.querySelector('video');
    
    if (videoElement) {
      console.log('[YouTube Dubbing] Video element found:', videoElement);
      initAttempts = 0;
    } else if (initAttempts < MAX_INIT_ATTEMPTS) {
      initAttempts++;
      console.log('[YouTube Dubbing] Video not found, retrying... attempt:', initAttempts);
      setTimeout(findVideoElement, 500);
    }
  }

  function observeNavigation() {
    // YouTube uses SPA navigation, so we need to detect URL changes
    let lastUrl = location.href;
    
    const observer = new MutationObserver(() => {
      if (location.href !== lastUrl) {
        lastUrl = location.href;
        onNavigate();
      }
    });
    
    observer.observe(document.body, { childList: true, subtree: true });
  }

  function onNavigate() {
    // Stop current dubbing on navigation
    stopDubbing();
    findVideoElement();
    console.log('[YouTube Dubbing] Navigation detected, reset state');
  }

  // Message listener
  chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
    switch (message.action) {
      case 'getVideoInfo':
        handleGetVideoInfo(sendResponse);
        break;
      
      case 'playDubbedAudio':
        handlePlayDubbedAudio(message, sendResponse);
        break;
      
      case 'stopDubbing':
        stopDubbing();
        sendResponse({ success: true });
        break;
      
      case 'updateVolume':
        updateOriginalVolume(message.volume);
        sendResponse({ success: true });
        break;
      
      default:
        sendResponse({ success: false, error: 'Unknown action' });
    }
    
    return true; // Keep channel open for async response
  });

  function handleGetVideoInfo(sendResponse) {
    try {
      // Try to find video element if not found yet
      if (!videoElement) {
        findVideoElement();
      }
      
      const videoId = getVideoId();
      
      if (!videoId) {
        console.log('[YouTube Dubbing] No video ID found in URL');
        sendResponse({ success: false, error: 'No video ID in URL' });
        return;
      }
      
      // Get title with multiple fallback selectors
      const title = getVideoTitle();
      
      console.log('[YouTube Dubbing] Video info:', { videoId, title, hasVideo: !!videoElement });
      
      sendResponse({
        success: true,
        videoId: videoId,
        title: title || 'YouTube Video',
        isDubbing: isDubbing,
        duration: videoElement ? videoElement.duration : 0
      });
    } catch (error) {
      console.error('[YouTube Dubbing] Get video info error:', error);
      sendResponse({ success: false, error: error.message });
    }
  }

  function getVideoTitle() {
    // Try multiple selectors for video title
    const selectors = [
      'h1.ytd-video-primary-info-renderer yt-formatted-string',
      'h1.ytd-watch-metadata yt-formatted-string',
      'h1.ytd-video-primary-info-renderer',
      'h1.ytd-watch-metadata',
      '#title h1',
      'yt-formatted-string.ytd-video-primary-info-renderer',
      '#container h1.title',
      'h1.title'
    ];
    
    for (const selector of selectors) {
      const element = document.querySelector(selector);
      if (element && element.textContent.trim()) {
        return element.textContent.trim();
      }
    }
    
    // Fallback to document title
    return document.title.replace(' - YouTube', '').trim();
  }

  function getVideoId() {
    const url = new URL(window.location.href);
    return url.searchParams.get('v');
  }

  async function handlePlayDubbedAudio(message, sendResponse) {
    try {
      segments = message.segments || [];
      originalVolume = (message.originalVolume || 20) / 100;
      
      // Initialize audio context
      if (!audioContext) {
        audioContext = new (window.AudioContext || window.webkitAudioContext)();
      }
      
      // Create gain node for dubbed audio volume control
      gainNode = audioContext.createGain();
      gainNode.gain.value = 1.0; // Full volume for dubbed audio
      gainNode.connect(audioContext.destination);
      
      // Lower original video volume
      if (videoElement) {
        videoElement.volume = originalVolume;
      }
      
      isDubbing = true;
      
      // Start playing dubbed segments synced with video
      startSegmentPlayback();
      
      // Add overlay indicator
      showDubbingIndicator();
      
      sendResponse({ success: true });
    } catch (error) {
      console.error('[YouTube Dubbing] Play dubbed audio error:', error);
      sendResponse({ success: false, error: error.message });
    }
  }

  function startSegmentPlayback() {
    if (!videoElement || segments.length === 0) return;
    
    // Clear any existing interval
    if (syncInterval) {
      clearInterval(syncInterval);
    }
    
    // Sync check every 100ms
    syncInterval = setInterval(() => {
      if (!isDubbing || !videoElement) {
        clearInterval(syncInterval);
        return;
      }
      
      const currentTime = videoElement.currentTime;
      
      // Find the appropriate segment for current time
      const segment = findSegmentForTime(currentTime);
      
      if (segment && segment.index !== currentSegmentIndex) {
        currentSegmentIndex = segment.index;
        playSegment(segment);
      }
      
      // Pause dubbed audio if video is paused
      if (videoElement.paused && sourceNode) {
        // Audio context doesn't have pause, we'll handle this differently
        gainNode.gain.value = 0;
      } else if (!videoElement.paused && gainNode) {
        gainNode.gain.value = 1.0;
      }
    }, 100);
    
    // Listen for video seek
    videoElement.addEventListener('seeked', onVideoSeek);
    videoElement.addEventListener('play', onVideoPlay);
    videoElement.addEventListener('pause', onVideoPause);
  }

  function findSegmentForTime(time) {
    for (let i = 0; i < segments.length; i++) {
      const seg = segments[i];
      if (time >= seg.start && time < seg.end) {
        return { ...seg, index: i };
      }
    }
    return null;
  }

  async function playSegment(segment) {
    if (!segment.audioUrl || !audioContext) return;
    
    try {
      // Stop current audio
      if (sourceNode) {
        sourceNode.stop();
        sourceNode.disconnect();
      }
      
      // Fetch and decode audio
      const response = await fetch(segment.audioUrl);
      const arrayBuffer = await response.arrayBuffer();
      const audioBuffer = await audioContext.decodeAudioData(arrayBuffer);
      
      // Create and play source
      sourceNode = audioContext.createBufferSource();
      sourceNode.buffer = audioBuffer;
      sourceNode.playbackRate.value = segment.speed || 1.0;
      sourceNode.connect(gainNode);
      
      // Calculate offset if we're mid-segment
      const currentTime = videoElement.currentTime;
      const offset = Math.max(0, currentTime - segment.start);
      
      sourceNode.start(0, offset);
      
      console.log(`[YouTube Dubbing] Playing segment ${segment.index} from ${offset.toFixed(2)}s`);
    } catch (error) {
      console.error('[YouTube Dubbing] Segment playback error:', error);
    }
  }

  function onVideoSeek() {
    currentSegmentIndex = -1; // Reset to force segment check
  }

  function onVideoPlay() {
    if (gainNode) {
      gainNode.gain.value = 1.0;
    }
  }

  function onVideoPause() {
    if (gainNode) {
      gainNode.gain.value = 0;
    }
  }

  function stopDubbing() {
    isDubbing = false;
    
    // Stop sync interval
    if (syncInterval) {
      clearInterval(syncInterval);
      syncInterval = null;
    }
    
    // Stop audio
    if (sourceNode) {
      try {
        sourceNode.stop();
        sourceNode.disconnect();
      } catch (e) {}
      sourceNode = null;
    }
    
    // Reset video volume
    if (videoElement) {
      videoElement.volume = 1.0;
      videoElement.removeEventListener('seeked', onVideoSeek);
      videoElement.removeEventListener('play', onVideoPlay);
      videoElement.removeEventListener('pause', onVideoPause);
    }
    
    // Remove indicator
    hideDubbingIndicator();
    
    // Reset state
    segments = [];
    currentSegmentIndex = 0;
    
    console.log('[YouTube Dubbing] Dubbing stopped');
  }

  function updateOriginalVolume(volume) {
    originalVolume = volume / 100;
    if (videoElement && isDubbing) {
      videoElement.volume = originalVolume;
    }
  }

  function showDubbingIndicator() {
    // Remove existing indicator
    hideDubbingIndicator();
    
    const indicator = document.createElement('div');
    indicator.id = 'yt-dubbing-indicator';
    indicator.innerHTML = `
      <style>
        #yt-dubbing-indicator {
          position: fixed;
          top: 80px;
          right: 20px;
          background: linear-gradient(135deg, #a855f7 0%, #7c3aed 100%);
          color: white;
          padding: 10px 16px;
          border-radius: 8px;
          font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
          font-size: 13px;
          font-weight: 500;
          z-index: 9999;
          display: flex;
          align-items: center;
          gap: 8px;
          box-shadow: 0 4px 12px rgba(0, 0, 0, 0.3);
          animation: slideIn 0.3s ease;
        }
        @keyframes slideIn {
          from { transform: translateX(100%); opacity: 0; }
          to { transform: translateX(0); opacity: 1; }
        }
        #yt-dubbing-indicator .pulse {
          width: 8px;
          height: 8px;
          background: #10b981;
          border-radius: 50%;
          animation: pulse 1.5s infinite;
        }
        @keyframes pulse {
          0%, 100% { opacity: 1; transform: scale(1); }
          50% { opacity: 0.7; transform: scale(1.2); }
        }
      </style>
      <span class="pulse"></span>
      <span>Dubbing Active - ToolWeb.in</span>
    `;
    
    document.body.appendChild(indicator);
  }

  function hideDubbingIndicator() {
    const indicator = document.getElementById('yt-dubbing-indicator');
    if (indicator) {
      indicator.remove();
    }
  }

})();
