// YouTube Dubbing Extension - Popup Script
// ToolWeb.in

const API_BASE_URL = 'https://api.toolweb.in:8161/dubby';

// DOM Elements
const elements = {
  videoStatus: document.getElementById('video-status'),
  statusDot: document.querySelector('.status-dot'),
  statusText: document.querySelector('.status-text'),
  videoTitle: document.getElementById('video-title'),
  targetLanguage: document.getElementById('target-language'),
  voiceGender: document.getElementById('voice-gender'),
  voiceSpeed: document.getElementById('voice-speed'),
  speedValue: document.getElementById('speed-value'),
  originalVolume: document.getElementById('original-volume'),
  volumeValue: document.getElementById('volume-value'),
  startButton: document.getElementById('start-dubbing'),
  stopButton: document.getElementById('stop-dubbing'),
  progressSection: document.getElementById('progress-section'),
  progressText: document.getElementById('progress-text'),
  progressPercent: document.getElementById('progress-percent'),
  progressFill: document.getElementById('progress-fill'),
  errorSection: document.getElementById('error-section'),
  errorMessage: document.getElementById('error-message')
};

// State
let currentVideoId = null;
let isDubbing = false;

// Initialize
document.addEventListener('DOMContentLoaded', init);

async function init() {
  // Load saved settings
  await loadSettings();
  
  // Setup event listeners
  setupEventListeners();
  
  // Check if we're on YouTube and get video info
  await checkCurrentTab();
}

async function loadSettings() {
  const settings = await chrome.storage.local.get([
    'targetLanguage',
    'voiceGender',
    'voiceSpeed',
    'originalVolume'
  ]);
  
  if (settings.targetLanguage) elements.targetLanguage.value = settings.targetLanguage;
  if (settings.voiceGender) elements.voiceGender.value = settings.voiceGender;
  if (settings.voiceSpeed) {
    elements.voiceSpeed.value = settings.voiceSpeed;
    elements.speedValue.textContent = settings.voiceSpeed + 'x';
  }
  if (settings.originalVolume !== undefined) {
    elements.originalVolume.value = settings.originalVolume;
    elements.volumeValue.textContent = settings.originalVolume + '%';
  }
}

function setupEventListeners() {
  // Settings changes
  elements.targetLanguage.addEventListener('change', saveSettings);
  elements.voiceGender.addEventListener('change', saveSettings);
  
  elements.voiceSpeed.addEventListener('input', () => {
    elements.speedValue.textContent = elements.voiceSpeed.value + 'x';
    saveSettings();
  });
  
  elements.originalVolume.addEventListener('input', () => {
    elements.volumeValue.textContent = elements.originalVolume.value + '%';
    saveSettings();
    // Send volume update to content script
    sendToContentScript({ action: 'updateVolume', volume: parseInt(elements.originalVolume.value) });
  });
  
  // Action buttons
  elements.startButton.addEventListener('click', startDubbing);
  elements.stopButton.addEventListener('click', stopDubbing);
}

async function saveSettings() {
  await chrome.storage.local.set({
    targetLanguage: elements.targetLanguage.value,
    voiceGender: elements.voiceGender.value,
    voiceSpeed: elements.voiceSpeed.value,
    originalVolume: elements.originalVolume.value
  });
}

async function checkCurrentTab() {
  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    
    if (!tab.url) {
      setStatus('error', 'Cannot access this page');
      return;
    }
    
    if (!tab.url.includes('youtube.com/watch')) {
      setStatus('not-youtube', 'Open a YouTube video to start');
      return;
    }
    
    // Extract video ID
    const url = new URL(tab.url);
    currentVideoId = url.searchParams.get('v');
    
    if (!currentVideoId) {
      setStatus('error', 'Could not detect video ID');
      return;
    }
    
    // Wait a moment for content script to initialize
    await new Promise(resolve => setTimeout(resolve, 500));
    
    // Get video info from content script with retry
    let response = null;
    let retries = 3;
    
    while (retries > 0 && (!response || !response.success)) {
      try {
        response = await sendToContentScript({ action: 'getVideoInfo' });
        if (response && response.success) break;
      } catch (e) {
        console.log('[Popup] Retry getting video info, attempts left:', retries - 1);
      }
      retries--;
      await new Promise(resolve => setTimeout(resolve, 500));
    }
    
    if (response && response.success) {
      setStatus('ready', 'Ready to dub');
      elements.videoTitle.textContent = response.title || 'Video detected';
      elements.startButton.disabled = false;
      
      // Check if already dubbing
      if (response.isDubbing) {
        isDubbing = true;
        elements.startButton.style.display = 'none';
        elements.stopButton.style.display = 'flex';
      }
    } else {
      // Even if content script fails, we have video ID from URL
      setStatus('ready', 'Ready to dub');
      elements.videoTitle.textContent = 'Video ID: ' + currentVideoId;
      elements.startButton.disabled = false;
    }
  } catch (error) {
    console.error('Tab check error:', error);
    setStatus('error', 'Extension error - try refreshing the page');
  }
}

function setStatus(type, message) {
  elements.statusText.textContent = message;
  elements.statusDot.className = 'status-dot';
  
  switch (type) {
    case 'ready':
      elements.statusDot.classList.add('active');
      break;
    case 'error':
    case 'not-youtube':
      elements.statusDot.classList.add('error');
      break;
    default:
      // Default pulsing animation
      break;
  }
}

async function startDubbing() {
  if (!currentVideoId) return;
  
  hideError();
  showProgress('Fetching captions...', 0);
  
  const settings = {
    videoId: currentVideoId,
    targetLanguage: elements.targetLanguage.value,
    voiceGender: elements.voiceGender.value,
    voiceSpeed: parseFloat(elements.voiceSpeed.value),
    originalVolume: parseInt(elements.originalVolume.value)
  };
  
  try {
    // Send request to background script
    const response = await chrome.runtime.sendMessage({
      action: 'startDubbing',
      settings: settings
    });
    
    if (response.success) {
      isDubbing = true;
      elements.startButton.style.display = 'none';
      elements.stopButton.style.display = 'flex';
      
      // Start polling for progress
      pollProgress(response.jobId);
    } else {
      showError(response.error || 'Failed to start dubbing');
      hideProgress();
    }
  } catch (error) {
    console.error('Start dubbing error:', error);
    showError('Failed to connect to server');
    hideProgress();
  }
}

async function stopDubbing() {
  try {
    await sendToContentScript({ action: 'stopDubbing' });
    
    isDubbing = false;
    elements.startButton.style.display = 'flex';
    elements.stopButton.style.display = 'none';
    hideProgress();
    setStatus('ready', 'Ready to dub');
  } catch (error) {
    console.error('Stop dubbing error:', error);
  }
}

async function pollProgress(jobId) {
  const pollInterval = setInterval(async () => {
    try {
      const response = await chrome.runtime.sendMessage({
        action: 'checkProgress',
        jobId: jobId
      });
      
      if (response.status === 'completed') {
        clearInterval(pollInterval);
        showProgress('Dubbing active!', 100);
        
        // Send dubbed audio to content script
        await sendToContentScript({
          action: 'playDubbedAudio',
          audioData: response.audioData,
          segments: response.segments,
          originalVolume: parseInt(elements.originalVolume.value)
        });
        
        setTimeout(() => {
          hideProgress();
          setStatus('ready', 'Dubbing active');
        }, 1000);
        
      } else if (response.status === 'error') {
        clearInterval(pollInterval);
        showError(response.error || 'Processing failed');
        hideProgress();
        isDubbing = false;
        elements.startButton.style.display = 'flex';
        elements.stopButton.style.display = 'none';
        
      } else if (response.status === 'processing') {
        updateProgress(response.progress, response.message);
      }
    } catch (error) {
      console.error('Poll error:', error);
    }
  }, 1000);
  
  // Store interval ID for cleanup
  chrome.storage.local.set({ pollIntervalId: pollInterval });
}

function showProgress(message, percent) {
  elements.progressSection.style.display = 'block';
  elements.progressText.textContent = message;
  elements.progressPercent.textContent = percent + '%';
  elements.progressFill.style.width = percent + '%';
}

function updateProgress(percent, message) {
  elements.progressText.textContent = message || 'Processing...';
  elements.progressPercent.textContent = Math.round(percent) + '%';
  elements.progressFill.style.width = percent + '%';
}

function hideProgress() {
  elements.progressSection.style.display = 'none';
}

function showError(message) {
  elements.errorSection.style.display = 'flex';
  elements.errorMessage.textContent = message;
}

function hideError() {
  elements.errorSection.style.display = 'none';
}

async function sendToContentScript(message) {
  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    if (tab && tab.id) {
      return await chrome.tabs.sendMessage(tab.id, message);
    }
  } catch (error) {
    console.error('Send to content script error:', error);
    return null;
  }
}

// Listen for messages from background/content scripts
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message.action === 'progressUpdate') {
    updateProgress(message.progress, message.message);
  } else if (message.action === 'dubbingComplete') {
    showProgress('Dubbing active!', 100);
    setTimeout(hideProgress, 2000);
  } else if (message.action === 'dubbingError') {
    showError(message.error);
    hideProgress();
    isDubbing = false;
    elements.startButton.style.display = 'flex';
    elements.stopButton.style.display = 'none';
  }
  sendResponse({ received: true });
  return true;
});
